/* el_syn.y -- Syntaxanalysator-Beschreibung fuer ELCOM */

%{

#include "elcom.h"

extern struct typliste *defs;                /* elcom.c */
extern struct ausdruck *prog;                /* elcom.c */

extern char fname[85];                       /* elcom.c */

int syntaxerror = 0;

char *fehler1 = "\"%s\" line %d: ";
char *fehler2 = "Constructor \"%s\" not ";
char *fehler3 = "found in type declaration \n";

%}

%union {
    struct typliste  *typzeiger;
    struct sumliste  *sumzeiger;
    struct prodliste *prodzeiger;
    struct ausdruck  *auszeiger;
    struct defliste  *defzeiger;
    char             *val;
}

%token _TYPE _END _LAMBDA _IF _THEN _ELSE _CASE _OF
%token _ARROW _FAIL _FATBAR _CONSTRUCT _SELECT _LET
%token _IN _LETREC

%token <val> _SCHEMVAR
%token <val> _CONSTRUCTOR
%token <val> _IDENTIFIER
%token <val> _INTEGER
%token <val> _REAL
%token <val> _CHAR
%token <val> _STRING

%type <typzeiger>  typen
%type <typzeiger>  typdefs
%type <typzeiger>  def
%type <prodzeiger> schemvars
%type <prodzeiger> schemvar
%type <sumzeiger>  sumtypen
%type <sumzeiger>  sum
%type <prodzeiger> prodtypen
%type <prodzeiger> prod

%type <auszeiger>  ausdr
%type <defzeiger>  vars
%type <defzeiger>  var
%type <defzeiger>  varliste
%type <auszeiger>  faelle
%type <auszeiger>  fall
%type <auszeiger>  ausdruecke
%type <defzeiger>  vardefs
%type <defzeiger>  vardef

%%

programm:    typen ausdr
                 { defs = $1; prog = $2; }
           | ausdr
                 { defs = NULL; prog = $1; }
           ;

typen:       _TYPE typdefs _END
                 { $$ = defs = $2; ueberpruefe($2); }
           ;

typdefs:     def
                 { $$ = $1; }
           | def ';' typdefs
                 { $$ = app_typ($1,$3); }
           ;

def:         _IDENTIFIER schemvars '=' sumtypen
                 { $$ = b_typ($1,$2,$4); }
           | error
                 { yyerror("Error in typ declaration"); }
           ;

schemvars:       { $$ = NULL; }
           | schemvar schemvars
                 { $$ = app_prod($1,$2); }
           ;

schemvar:    _SCHEMVAR
                 { $$ = b_prod($1,NULL); }
           | error
                 { yyerror("Schemat. var. expected"); }
           ;

sumtypen:    sum
                 { $$ = $1; }
           | sum '|' sumtypen
                 { $$ = app_sum($1,$3); }
           ;

sum:         _CONSTRUCTOR prodtypen
                 { $$ = b_sum($1,$2); }
           | error
                 { yyerror("Constructor + args expected"); }
           ;

prodtypen:       { $$ = NULL; }
           | prod prodtypen
                 { $$ = app_prod($1,$2); }

prod:        _IDENTIFIER
                 { $$ = b_prod($1,NULL); }
           | schemvar
                 { $$ = $1; }
           | '(' _IDENTIFIER prodtypen ')'
                 { $$ = b_prod($2,$3); }
           ;


ausdr:       _INTEGER
                 { $$ = b_konst(KONST,INTEGER,$1); }
           | _REAL
                 { $$ = b_konst(KONST,REAL,$1); }
           | _CHAR
                 { $$ = b_konst(KONST,CHAR,$1); }
           | _STRING
                 { $$ = b_string($1); }
           | _FAIL
                 { $$ = b_konst(FAIL,0,NULL); }
           | _IDENTIFIER
                 { $$ = b_konst(VAR,VARNAME,$1); }
           | '(' ausdr ausdr ')'
                 { $$ = b_anw(ANW,$2,$3); }
           | _FATBAR '(' ausdr ausdr ')'
                 { $$ = b_anw(FATBAR,$3,$4); }
           | _LAMBDA vars '.' ausdr _END
                 { $$ = b_lambda($2,$4); }
           | _IF ausdr _THEN ausdr _ELSE ausdr _END
                 { $$ = b_if($2,$4,$6); }
           | _CASE ausdr _OF faelle _END
                 { $$ = b_case($2,$4); }
           | _CONSTRUCT '(' _CONSTRUCTOR ausdruecke ')'
                 { $$ = b_cons($3,$4); }
           | _SELECT '(' _INTEGER ',' ausdr ')'
                 { $$ = b_sel($3,$5); }
           | _LET vardef _IN ausdr _END
                 { $$ = b_let(LET,$2,$4); }
           | _LETREC vardefs _IN ausdr _END
                 { $$ = b_let(LETREC,$2,$4); }
           | error
                 { yyerror("Error in expression"); }
           ;

vars:        var
                 { $$ = $1; }
           | var vars
                 { $$ = app_def($1,$2); }
           ;

var:         _IDENTIFIER
                 { $$ = b_def($1,NULL); }
           | error
                 { yyerror("Identifier expected"); }
           ;

faelle:      fall
                 { $$ = $1; }
           | fall ';' faelle
                 { $$ = app_ausl($1,$3); }
           ;

fall:        '(' _CONSTRUCTOR varliste ')' _ARROW ausdr
                 { $$ = b_fall($2,$3,$6); }
           | _CONSTRUCTOR varliste _ARROW ausdr
                 { $$ = b_fall($1,$2,$4); }
           | error
                 { yyerror("Pattern => expr. expected"); }
           ;

varliste:        { $$ = NULL; }
           | vars
                 { $$ = $1; }
           ;

ausdruecke:      { $$ = NULL; }
           | ',' ausdr ausdruecke
                 { $$ = app_ausl($2,$3); }
           ;

vardefs:     vardef
                 { $$ = $1; }
           | vardef ';' vardefs
                 { $$ = app_def($1,$3); }
           ;

vardef:      _IDENTIFIER '=' ausdr
                 { $$ = b_def($1,$3); }
           | error
                 { yyerror("Identifier = expr. expected"); }
           ;

%%

#include "lex.yy.c"

struct typliste *app_typ (p, q)
struct typliste *p, *q;
{
    struct typliste *r;

    r = p;
    if (r == NULL)
        return q;
    while (r->next != NULL)
        r = r->next;
    r->next = q;
    return p;
} /* app_typ */

int len_sum (p)
struct sumliste *p;
{
    int i;

    i = 0;
    while (p != NULL) {
        i++;
        p = p->next;
    }
    return i;
} /* len_sum */

int len_prod (p)
struct prodliste *p;
{
    int i;

    i = 0;
    while (p != NULL) {
        i++;
        p = p->next;
    }
    return i;
} /* len_prod */

struct typliste *b_typ (op, var, sum)
char             *op;
struct prodliste *var;
struct sumliste  *sum;
{
    struct typliste *r;

    r = (struct typliste *) new(TYPLISTE);
    r->zeile = yylineno;
    r->typop = op;
    r->konstanz = len_sum(sum);
    r->schemanz = len_prod(var);
    r->vars = var;
    r->typen = sum;
    r->next = NULL;
    return r;
} /* b_typ */

struct sumliste *app_sum (p, q)
struct sumliste *p, *q;
{
    struct sumliste *r;

    r = p;
    if (r == NULL)
        return q;
    while (r->next != NULL)
        r = r->next;
    r->next = q;
    return p;
} /* app_sum */

struct sumliste *b_sum (k, prod)
char             *k;
struct prodliste *prod;
{
    struct sumliste *p;

    p = (struct sumliste *) new(SUMLISTE);
    p->begzeile = yylineno;
    p->kons = k;
    p->stell = len_prod(prod);
    p->prodtypen = prod;
    p->next = NULL;
    return p;
} /* b_sum */

struct prodliste *app_prod (p, q)
struct prodliste *p, *q;
{
    struct prodliste *r;

    r = p;
    if (r == NULL)
        return q;
    while (r->next != NULL)
        r = r->next;
    r->next = q;
    return p;
} /* app_prod */

struct prodliste *b_prod (c, p)
char             *c;
struct prodliste *p;
{
    struct prodliste *r;

    r = (struct prodliste *) new(PRODLISTE);
    r->name = c;
    r->nr = 0;
    r->typ = NULL;
    r->belegung = p;
    r->next = NULL;
    return r;
} /* b_prod */


struct ausdruck *b_konst(ausart,typ,konstwert)
int  ausart,typ;
char *konstwert;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = ausart;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    p->standardtyp = typ;
    p->wert = konstwert;
    p->links = p->rechts = p->hinten = NULL;
    return p;
} /* b_konst */

struct ausdruck *b_anw (ausart, aus1, aus2)
int             ausart;
struct ausdruck *aus1, *aus2;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = ausart;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    p->standardtyp = 0;
    p->wert = NULL;
    p->links = aus1;
    p->rechts = aus2;
    p->hinten = NULL;
    return p;
} /* b_anw */

int len_def (p)
struct defliste *p;
{
    int i;

    i = 0;
    while (p != NULL) {
        i++;
        p = p->next;
    }
    return i;
} /* len_def */

struct ausdruck *b_lambda(vars, aus)
struct defliste *vars;
struct ausdruck *aus;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = LAMBDA;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = vars;
    p->stelligkeit = len_def(vars);
    p->standardtyp = 0;
    p->wert = NULL;
    p->links = aus;
    p->rechts = p->hinten = NULL;
    return p;
} /* b_lambda */

struct ausdruck *b_if(aus1, aus2, aus3)
struct ausdruck *aus1, *aus2, *aus3;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = IF;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    p->standardtyp = 0;
    p->wert = NULL;
    p->links = aus1;
    p->rechts = aus2;
    aus2->hinten = aus3;
    p->hinten = NULL;
    return p;
} /* b_if */

int findekonst (t, c)
struct typliste *t;
char            *c;
{
    struct sumliste *p;
    int             i;

    p = t->typen;
    i = 1;
    while (p != NULL & strcmp(p->kons,c)) {
        i++;
        p = p->next;
    }
    if (p != NULL)
        return i;
    return 0;
} /* findekonst */

struct typliste *findetyp (c)
char *c;
{
    struct typliste *t;

    t = defs;
    while (t != NULL) {
        if (findekonst(t,c))
            break;
        t = t->next;
    }
    return t;
} /* findetyp */

struct ausdruck *b_case (aus1, aus2)
struct ausdruck *aus1, *aus2;
{
    struct typliste *t;
    struct ausdruck *a, *p, *pp;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = CASE;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    p->standardtyp = 0;
    p->wert = NULL;
    p->links = aus1;
    p->rechts = aus2;
    p->hinten = NULL;
    a = p->rechts;
    if ((t = findetyp(a->wert)) == NULL) {
        fprintf(stderr,fehler1,fname,a->beginn);
        fprintf(stderr,fehler2,a->wert);
        fputs(fehler3,stderr);
        syntaxerror = 1;
        return p;
    }
    while (a != NULL) {
        if ((a->standardtyp = findekonst(t,a->wert))
            == 0) {
            fprintf(stderr,fehler1,fname,a->beginn);
            fprintf(stderr,fehler2,a->wert);
            fputs("defined in type \"",stderr);
            fputs(t->typop,stderr);
            fputs("\" \n",stderr);
            syntaxerror = 1;
        }
        pp = p->rechts;
        while (pp != NULL &&
               pp->standardtyp != a->standardtyp)
            pp = pp->hinten;
        if (pp != NULL && pp != a) {
            fprintf(stderr,fehler1,fname,a->beginn);
            fputs("Constructor \"",stderr);
            fputs(a->wert,stderr);
            fputs("\" occurs twice in CASE \n",stderr);
            syntaxerror = 1;
        }
        a = a->hinten;
    }
    return p;
} /* b_case */

struct ausdruck *b_cons (name, aus)
char            *name;
struct ausdruck *aus;
{
    struct typliste *t;
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = CONS;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    if ((t = findetyp(name)) == NULL) {
        fprintf(stderr,fehler1,fname,yylineno);
        fprintf(stderr,fehler2,name);
        fputs(fehler3,stderr);
        syntaxerror = 1;
    }
    else
        p->standardtyp = findekonst(t,name);
    p->wert = name;
    p->links = aus;
    p->rechts = p->hinten = NULL;
    return p;
} /* b_cons */

struct ausdruck *b_sel (index, aus)
char            *index;
struct ausdruck *aus;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = SELECT;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = NULL;
    p->stelligkeit = 0;
    p->standardtyp = INTEGER;
    p->wert = index;
    p->links = aus;
    p->rechts = p->hinten = NULL;
    return p;
} /* b_sel */

struct ausdruck *b_let (art, defs, aus)
int             art;
struct defliste *defs;
struct ausdruck *aus;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = art;
    p->beginn = yylineno;
    p->strikt = 0;
    p->definition = defs;
    p->stelligkeit = len_def(defs);
    p->standardtyp = 0;
    p->wert = NULL;
    p->links = aus;
    p->rechts = p->hinten = NULL;
    return p;
} /* b_let */

struct ausdruck *b_fall (kons, vars, aus)
char            *kons;
struct defliste *vars;
struct ausdruck *aus;
{
    struct ausdruck *p;

    p = (struct ausdruck *) new(AUSDRUCK);
    p->art = MUSTER;
    p->beginn = yylineno;
    p->definition = vars;
    p->stelligkeit = len_def(vars);
    p->standardtyp = -1;
    p->wert = kons;
    p->links = aus;
    p->rechts = p->hinten = NULL;
    return p;
} /* b_fall */

struct defliste *app_def (p, q)
struct defliste *p, *q;
{
    struct defliste *r;

    r = p;
    if (r == NULL)
        return q;
    while (r->next != NULL)
        r = r->next;
    r->next = q;
    return p;
} /* app_def */

struct defliste *b_def (var, wert)
char            *var;
struct ausdruck *wert;
{
    struct defliste *p;

    p = (struct defliste *) new(DEFLISTE);
    p->varname = var;
    p->defstrikt = 0;
    p->sym = NULL;
    p->varwert = wert;
    p->abst = NULL;
    p->next = NULL;
    return p;
} /* b_def */

struct ausdruck *app_ausl (p, q)
struct ausdruck *p, *q;
{
    struct ausdruck *r;

    r = p;
    if (r == NULL)
        return q;
    while (r->hinten != NULL)
        r = r->hinten;
    r->hinten = q;
    return p;
} /* app_ausl */

struct ausdruck *b_str (s)
char *s;
{
    struct ausdruck *p, *pp;
    char            *c;

    if (*s == '\0')
        return b_cons("NIL",NULL);
    c = new(4);
    if (*s == '\\')
        s++;
    c[0] = '\'';
    c[1] = *s;
    c[2] = '\'';
    c[3] = '\0';
    p = b_konst(KONST,CHAR,c);
    pp = b_str(s+1);
    return b_cons("CONS",app_ausl(p,pp));
} /* b_str */

struct ausdruck *b_string (s)
char *s;
{
    struct ausdruck *p;

    s[strlen(s)-1] = '\0';
    p = b_str(s+1);
    free(s);
    return p;
} /* b_string */

fehler (zeile, name)
int  zeile;
char *name;
{
    fprintf(stderr,fehler1,fname,zeile);
    fprintf(stderr,"Type/constructor \"%s\" ",name);
    fputs("declared twice \n",stderr);
    syntaxerror = 1;
} /* fehler */

struct defliste *nachsehen (zeile, name, p)
int             zeile;
char            *name;
struct defliste *p;
{
    struct defliste *r;

    r = p;
    while (r != NULL && strcmp(r->varname,name))
        r = r->next;
    if (r != NULL) {
        fehler(zeile,name);
        return p;
    }
    r = (struct defliste *) new(DEFLISTE);
    r->varname = name;
    r->next = p;
    return r;
} /* nachsehen */

int ueberpruefe (typen)
struct typliste *typen;
{
    struct defliste *p, *q;
    struct typliste *t;
    struct sumliste *s;

    p = NULL;
    t = typen;
    while (t != NULL) {
        p = nachsehen(t->zeile,t->typop,p);
        s = t->typen;
        while (s != NULL) {
            p = nachsehen(s->begzeile,s->kons,p);
            s = s->next;
        }
        t = t->next;
    }
    while (p != NULL) {
        q = p->next;
        free(p);
        p = q;
    }
} /* ueberpruefe */

int analyse (f)
FILE *f;
{
    yyin = f;
    if (yyparse() != 0)
        return 1;
    return syntaxerror;
} /* analyse */

yywrap ()
{
    return 1;
} /* yywrap */

yyerror (s)
char *s;
{
    fprintf(stderr,fehler1,fname,yylineno);
    fputs(s,stderr);
    putc('\n',stderr);
    syntaxerror = 1;
} /* yyerror */
