/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: ContentsPreviewManager
    Created: 17.02.2006 (18:45:27)
        $Id: ContentsPreviewManager.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel.contents;

import java.io.File;
import java.net.URL;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JEditorPane;
import javax.swing.JOptionPane;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.event.HyperlinkEvent.EventType;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.settings.Settings;
import com.dgrossmann.photo.ui.dialog.ExportedImageViewDialog;
import com.dgrossmann.photo.webexport.ExportFactory;

/**
 * Generates and manages the HTML contents preview for a directory object.
 * @author Dirk Grossmann
 */
public class ContentsPreviewManager implements HyperlinkListener
{
    public static final String PREVIEW_MAIN_IMAGES =
        "contents.preview.main_images";

    private static final String TAG_SUB_GROUP_LIST = "<sub-group-list>";
    private static final String TAG_FILE_LIST      = "<file-list>";

    /**
     * Constant for the number of index image columns to be displayed in the
     * HTML view.
     */
    private static final int NUM_INDEX_COLS = 3;

    /**
     * Constant for the number of index image columns to be displayed in the
     * HTML view.
     */
    private static final int NUM_MAIN_COLS = 2;

    private ContentsPanel                 m_parentPanel;
    private JEditorPane                   m_htmlView;
    private DirectoryObject               m_currentDirectory;
    private boolean                       m_bIsActive;
    private boolean                       m_bNeedRefresh;
    private boolean                       m_bShowMainImages;
    private Map<Object, AbstractFSObject> m_imageMap;

    /**
     * Creates a new <tt>ContentsPreviewManager</tt> instance.
     * @param parentPanel - The contents panel
     * @param htmlView - The HTML view
     */
    public ContentsPreviewManager
        ( ContentsPanel parentPanel
        , JEditorPane   htmlView
        )
    {
        m_parentPanel = parentPanel;
        m_htmlView = htmlView;
        m_currentDirectory = null;
        m_bIsActive = false;
        m_bNeedRefresh = false;
        m_bShowMainImages = false;
        m_imageMap = new HashMap<Object, AbstractFSObject>();
        m_htmlView.setContentType("text/html");
        m_htmlView.setEditable(false);
        m_htmlView.addHyperlinkListener(this);
    } // ContentsPreviewManager

    /**
     * Sets the current directory.
     * @param currentDir - The new current directory
     */
    public void setCurrentDirectory (DirectoryObject currentDir)
    {
        m_currentDirectory = currentDir;
        m_bNeedRefresh = true;
        this.refresh();
    } // setCurrentDirectory

    /**
     * Activates this instance. This updates the HTML view as necessary.
     * @param bActive - <tt>True</tt> to activate, <tt>false</tt> to inactivate
     */
    public void setActive (boolean bActive)
    {
        m_bIsActive = bActive;
        if (m_bIsActive)
        {
            m_bNeedRefresh = true;
            this.refresh();
        }
    } // setActive

    //--------------------------------------------------------------------------
    // HTML display methods
    //==========================================================================

    /**
     * Refreshes this instance.
     */
    public void refresh ()
    {
        URL                  styleSheetURL;
        StringBuffer         html;
        Iterator<FileObject> iter;
        FileObject           fileObj;
        boolean              bHaveExportedFiles;

        if (!m_bIsActive || !m_bNeedRefresh)
            return;
        m_bNeedRefresh = false;
        // Clear the view on null directory.
        if (m_currentDirectory == null)
        {
            m_htmlView.setText(this.showOff());
            m_htmlView.select(0, -1);
            return;
        }
        // Generate the HTML.
        styleSheetURL = this.getClass().getClassLoader().
            getResource("images/styles.css");
        html = new StringBuffer(10000);
        html.append("<html><head><title>HTML Preview</title>\n");
        if (styleSheetURL != null)
        {
            html.append("<link rel='stylesheet' type='text/css' href='").
                append(styleSheetURL.toExternalForm()).append("'>\n");
        }
        html.append("</head>\n").append("<body>\n");
        html.append("<h1><font color='#999999'>Contents of ");
        if (m_currentDirectory.getParent() == null)
            html.append("series");
        else
            html.append("group");
        html.append("</font> <font color='#D32C58'>").
            append(m_currentDirectory.getTitle(true)).
            append("</font></h1>\n");
        // Show the link to switch index and main images.
        bHaveExportedFiles = false;
        iter = m_currentDirectory.getFileIterator();
        while (iter.hasNext())
        {
            fileObj = iter.next();
            if (fileObj.isToExport())
            {
                bHaveExportedFiles = true;
                break;
            }
        }
        if (bHaveExportedFiles)
        {
            html.append("<p><font color='#999999'>");
            if (m_bShowMainImages)
            {
                html.append("The following shows all main images.</font>").
                    append(" <a href='#index'><b>Show index images</b>.</a>");
            }
            else
            {
                html.append("The following shows all index images.</font>").
                    append(" <a href='#main'><b>Show main images</b>.</a>");
            }
            html.append("</p>\n");
        }
        // Check whether we have all exported files. If not, do the export.
        // ...
        // Show the directories and files.
        m_imageMap.clear();
        if (this.showDirOverview(html))
            this.showAllFiles(html);
        if (!bHaveExportedFiles)
        {
            html.append("<p>&nbsp;</p><h2><font color='#999999'>There are ").
                append("no exported files in this group.</font></h2>\n");
        }
        // Close the HTML and display it in the HTML view.
        html.append("</body></html>");
        m_htmlView.setText(html.toString());
        m_htmlView.select(0, -1);
    } // refresh

    private static final int SHOW_NOTHING = 0;
    private static final int SHOW_SUBDIRS = 1;
    private static final int SHOW_FILES   = 2;

    /**
     * Private method to show the directory overview including the
     * subdirectories in the HTML view.
     * @param html - String buffer to append the contents to
     * @return <tt>True</tt> if the files should be displayed explicitly;
     * <tt>false</tt> if this method has already displayed the files
     */
    private boolean showDirOverview (StringBuffer html)
    {
        String  desc, descBegin, descMiddle, descEnd, repl;
        int     posSubGroups, posFiles, pos1, len1, pos2, len2;
        int     firstOp, secondOp;
        boolean bShowSubDirs, bShowFiles;

        if (m_currentDirectory == null)
            return false;
        bShowSubDirs = bShowFiles = true;
        desc = m_currentDirectory.get(AbstractFSObject.DESCRIPTION);
        if (desc != null && desc.length() > 0)
        {
            descMiddle = descEnd = "";
            firstOp = secondOp = SHOW_NOTHING;
            // Prepend <img src=''> elements with the web export directory name.
            desc = this.processImgElements(desc);
            // Get the position of the tags and partition the string.
            posSubGroups = desc.indexOf(TAG_SUB_GROUP_LIST);
            posFiles = desc.indexOf(TAG_FILE_LIST);
            if (posSubGroups < 0 && posFiles < 0)
                descBegin = desc;
            else if (posFiles < 0)
            {
                descBegin = desc.substring(0, posSubGroups).trim();
                descMiddle = desc.substring(posSubGroups +
                    TAG_SUB_GROUP_LIST.length());
                firstOp = SHOW_SUBDIRS;
            }
            else if (posSubGroups < 0)
            {
                descBegin = desc.substring(0, posFiles).trim();
                descMiddle = desc.substring(posFiles + TAG_FILE_LIST.length()).
                    trim();
                firstOp = SHOW_FILES;
            }
            else
            {
                if (posSubGroups < posFiles)
                {
                    pos1 = posSubGroups;
                    len1 = TAG_SUB_GROUP_LIST.length();
                    pos2 = posFiles;
                    len2 = TAG_FILE_LIST.length();
                    firstOp = SHOW_SUBDIRS;
                    secondOp = SHOW_FILES;
                }
                else
                {
                    pos2 = posSubGroups;
                    len2 = TAG_SUB_GROUP_LIST.length();
                    pos1 = posFiles;
                    len1 = TAG_FILE_LIST.length();
                    firstOp = SHOW_FILES;
                    secondOp = SHOW_SUBDIRS;
                }
                descBegin = desc.substring(0, pos1).trim();
                descMiddle = desc.substring(pos1+len1, pos2-pos1-len1).trim();
                descEnd = desc.substring(pos2+len2).trim();
            }
            // Print the strings.
            repl = "</p>\n<p>";
            html.append("<p>");
            html.append(descBegin.replaceAll("\n\n", repl)).append("\n");
            if (firstOp == SHOW_SUBDIRS)
            {
                this.showAllSubDirectories(false, html);
                bShowSubDirs = false;
            }
            else if (firstOp == SHOW_FILES)
            {
                this.showAllFiles(html);
                bShowFiles = false;
            }
            if (descMiddle.length() > 0)
                html.append(descMiddle.replaceAll("\n\n", repl)).append("\n");
            if (secondOp == SHOW_SUBDIRS)
            {
                this.showAllSubDirectories(false, html);
                bShowSubDirs = false;
            }
            else if (secondOp == SHOW_FILES)
            {
                this.showAllFiles(html);
                bShowFiles = false;
            }
            if (descEnd.length() > 0)
                html.append(descEnd.replaceAll("\n\n", repl)).append("\n");
            html.append("</p>\n");
        }
        if (bShowSubDirs)
            this.showAllSubDirectories(true, html);
        return bShowFiles;
    } // showDirOverview

    /**
	 * Private method that prepends &lt;img src=''&gt; elements with the web
	 * export directory name.
	 * @param descStr - The string
	 * @return String with the expanded elements
	 */
    private String processImgElements (String descStr)
    {
        StringBuffer sb;
        URL          exportDirUrl;
        int          i, index;

        try
        {
            exportDirUrl = (new File(m_parentPanel.getFrame().getSettings().
                get(Settings.EXPORT_DIRECTORY))).toURI().toURL();
        }
        catch (Exception exc)
        {
            return descStr;
        }
        sb = new StringBuffer(descStr.length() + 222);
        i = 0;
        while ((index = descStr.indexOf("src=", i)) >= 0)
        {
            // Append the begin part to the string buffer (including src=).
            index += 4;
            sb.append(descStr.substring(i, index));
            i = index;
            while (i < descStr.length() && (descStr.charAt(i) == ' ' ||
                   descStr.charAt(i) == '\t'))
                i++;
            sb.append(descStr.charAt(i++));
            if (descStr.charAt(i) != '/')
                sb.append(exportDirUrl.toExternalForm()).append('/');
        }
        // Append the rest and return.
        if (i == 0)
            return descStr;
        sb.append(descStr.substring(i));
        return sb.toString();
    } // processImgElements

    /**
     * Private method to show all subdirectories
     * @param bWithHeading - <tt>True</tt> to include a heading if we have sub
     * directories to show; <tt>false</tt> no never include a heading
     * @param html - String buffer to append the contents to
     */
    private void showAllSubDirectories (boolean bWithHeading, StringBuffer html)
    {
        StringBuffer              sb;
        URL                       arrowURL;
        Iterator<DirectoryObject> iter;
        DirectoryObject           dirObj;
        String                    link;
        boolean                   bHaveDir;

        if (m_currentDirectory == null ||
            m_currentDirectory.getSubDirCount() == 0)
        {
            return;
        }
        bHaveDir = false;
        arrowURL = this.getClass().getClassLoader().
            getResource("images/nav-arrow.gif");
        sb = new StringBuffer(1000);
        sb.append("<p>\n");
        iter = m_currentDirectory.getSubDirIterator();
        while (iter.hasNext())
        {
            dirObj = iter.next();
            if (!dirObj.isToExport())
                continue;
            sb.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<img src='").
                append(arrowURL.toExternalForm()).append("'>");
            link = "#dir:" + dirObj.getFileName();
            m_imageMap.put(link, dirObj);
            sb.append("&nbsp;&nbsp;<a href='").append(link).append("'><b>").
                append(dirObj.getTitle(true)).append("</b></a><br>\n");
            bHaveDir = true;
        }
        sb.append("</p>\n");
        if (bHaveDir)
        {
            if (bWithHeading)
                html.append("<h2>Sub Groups</h2>\n");
            html.append(sb);
        }
    } // showAllSubDirectories

    /**
     * Private method to show the exported files contained in the current
     * directory in the HTML view.
     * @param html - String buffer to append the contents to
     */
    private void showAllFiles (StringBuffer html)
    {
    	Iterator<FileObject> iter;
        FileObject           fileObj;
        String               tag;
        int                  numColumns, i;
        boolean              bIsFirstRow, bNewRowBefore;

        i = 0;
        bIsFirstRow = true;
        bNewRowBefore = false;
        if (m_bShowMainImages)
        	numColumns = (m_htmlView.getWidth() >= 1000) ? NUM_MAIN_COLS : 1;
        else
        	numColumns = NUM_INDEX_COLS;
        // Print the file table start.
        html.append("<table border='0' cellpadding='8' cellspacing='8'>\n");
        iter = m_currentDirectory.getFileIterator();
        while (iter.hasNext())
        {
            fileObj = iter.next();
            if (!fileObj.isToExport())
                continue;
            // Force a new row for separators and references (unless it ends).
            if (fileObj.isSeparator() || fileObj.isReference())
            {
                if (i % numColumns != 0)
                {
                    bNewRowBefore = true;
                    if (!bIsFirstRow)
                        html.append(" </tr>\n");
                }
            }
            // Build the file object tag.
            try
            {
                tag = this.showOneFile(fileObj);
            }
            catch (Exception exc)
            {
                tag = exc.toString();
            }
            // Print a new row start.
            if (bIsFirstRow || bNewRowBefore || i % numColumns == 0)
            {
                html.append(" <tr>\n");
                bNewRowBefore = false;
            }
            // Print the image/reference tag.
            if (fileObj.isReference())
                html.append("  <td colspan='" + numColumns + "'>" + tag
                    + "</td>\n");
            else
            {
                html.append("  <td width='33%' valign='bottom' nowrap>"
                    + tag + "</td>\n");
            }
            // Forward to the next image.
            i++;
            if (fileObj.isReference())
                i = numColumns;
            // Print the row end.
            if (i % numColumns == 0 || !iter.hasNext())
                html.append(" </tr>\n");
            bIsFirstRow = false;
        }
        // Print the file table end.
        html.append("</table>\n\n");
    } // showAllFiles

    /**
     * Private method to show one file.
     * @param fileObj - The file to show
     * @return The HTML string
     */
    private String showOneFile (FileObject fileObj)
        throws Exception
    {
        File[] exportedFiles;
        URL    mainFileUrl, indexFileUrl;
        String tag, mainSize, indexSize;
        long   len;

        // Get the tag for separators and references.
        if (fileObj.isSeparator())
            return "<hr noshade size='1' width='22%' color='#666666'>";
        if (fileObj.isReference())
        {
            tag = "<p>";
            if (fileObj.get(AbstractFSObject.HREF).length() > 0)
            {
                tag += "<a href='" + fileObj.get(AbstractFSObject.HREF)
                    + "'><b>" + fileObj.getTitle(true) + "</b></a><br>\n";
            }
            return tag + fileObj.get(AbstractFSObject.DESCRIPTION) + "</p>\n";
        }
        if (fileObj.getFileType() != FileObject.TYPE_IMAGE_PREVIEW)
        {
            return "<p><b>" + fileObj.getTitle(true) + "</b><br>\n"
                + fileObj.get(AbstractFSObject.DESCRIPTION) + "</p>\n";
        }
        // Get the image sources.
        exportedFiles = ExportFactory.getExport(
            m_parentPanel.getFrame().getSettings(),
            m_parentPanel.getFrame().getSeriesContainer(), m_parentPanel).
            getExportedFiles(fileObj);
        if (exportedFiles == null || exportedFiles.length < 2 ||
        	exportedFiles[0] == null || exportedFiles[1] == null)
        {
            return "<p><b>" + fileObj.getTitle(true)
                + "</b><br>There are no exported files.</p>\n";
        }
        mainFileUrl = exportedFiles[0].toURI().toURL();
        indexFileUrl = (m_bShowMainImages) ?
            mainFileUrl : exportedFiles[1].toURI().toURL();
        m_imageMap.put(mainFileUrl, fileObj);
        // Get the file sizes.
        mainSize = indexSize = "--";
        if (exportedFiles[0] != null)
        {
            len = exportedFiles[0].length() / 1024;
            if (len > 0)
                mainSize = Long.toString(len) + " KB";
        }
        if (exportedFiles[1] != null)
        {
            len = exportedFiles[1].length() / 1024;
            if (len > 0)
                indexSize = Long.toString(len) + " KB";
        }
        // Get the tag.
        return "<a href='" + mainFileUrl.toExternalForm() + "'><img src='"
            + indexFileUrl.toExternalForm() + "' border='0'></a><br>\n"
            + "<a href='" + mainFileUrl.toExternalForm() + "'><b>"
            + fileObj.getTitle(true) + "</b></a>"
            + (m_bShowMainImages ? " &nbsp; " : "<br>")
            + "(Main: " + mainSize + ", Index: " + indexSize + ")\n";
    } // showOneFile

    /**
     * @see javax.swing.event.HyperlinkListener#hyperlinkUpdate(javax.swing.event.HyperlinkEvent)
     */
    public void hyperlinkUpdate (HyperlinkEvent e)
    {
        String          desc;
        DirectoryObject dirObj;
        FileObject      fileObj;

        if (e.getEventType() != EventType.ACTIVATED)
            return;
        // Check for image size switch.
        desc = e.getDescription();
        if ("#main".equals(desc))
        {
            m_bShowMainImages = true;
            m_bNeedRefresh = true;
            this.refresh();
            return;
        }
        if ("#index".equals(desc))
        {
            m_bShowMainImages = false;
            m_bNeedRefresh = true;
            this.refresh();
            return;
        }
        if (desc != null && desc.startsWith("#dir:"))
        {
            dirObj = (DirectoryObject) m_imageMap.get(desc);
            if (dirObj != null)
                m_parentPanel.fireCurrentDirectoryChanged(dirObj);
            return;
        }
        // A file should be displayed.
        fileObj = (FileObject) m_imageMap.get(e.getURL());
        if (fileObj == null)
        {
            JOptionPane.showMessageDialog(m_parentPanel,
                AppInfo.APP_NAME + " cannot follow this link:\n" + desc,
                AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
            return;
        }
        ExportedImageViewDialog dlg = new ExportedImageViewDialog(fileObj,
            m_parentPanel.getFrame().getSettings(),
            m_parentPanel.getFrame().getSeriesContainer(),
            m_parentPanel.getFrame(), true);
        dlg.setVisible(true);
    } // hyperlinkUpdate

    /**
     * Loads the preview settings.
     * @param settings - The settings object
     */
    public void loadSettings (Settings settings)
    {
        m_bShowMainImages = settings.getBoolean(PREVIEW_MAIN_IMAGES, false);
        m_bNeedRefresh = true;
    } // loadSettings

    /**
     * Saves the preview settings.
     * @param settings - The settings object
     */
    public void saveSettings (Settings settings)
    {
        settings.setBoolean(PREVIEW_MAIN_IMAGES, m_bShowMainImages);
    } // saveSettings

    /**
     * Private method to get the default HTML view text for a <tt>null</tt>
     * current directory.
     * @return The default HTML view text.
     */
    private String showOff ()
    {
        StringBuffer html;
        URL          styleSheetURL, appLogoURL;

        html = new StringBuffer(900);
        styleSheetURL = this.getClass().getClassLoader().getResource(
            "images/styles.css");
        appLogoURL = this.getClass().getClassLoader().
            getResource("images/jp-logo.jpg");
        html = new StringBuffer(10000);
        html.append("<html><head><title>HTML Preview</title>\n");
        if (styleSheetURL != null)
        {
            html.append("<link rel='stylesheet' type='text/css' href='")
                .append(styleSheetURL.toExternalForm()).append("'>\n");
        }
        html.append("</head>\n").append("<body>\n");
        html.append("<table border='0' cellpadding='0' cellspacing='0' ").
            append("width='95%'>\n<tr><td height='95%' align='center'>\n");
        html.append("<table border='0' cellpadding='0' cellspacing='0'>\n").
        append("<tr><td height='95%'>\n");
        html.append("<p></p>\n");
        html.append("<h1>").append(AppInfo.APP_NAME).append("</h1>\n");
        html.append("<p><b>Version ").append(AppInfo.getVersionString()).
            append("</b><br>").append(AppInfo.getBuildDate(false)).
            append("</b></p>");
        if (appLogoURL != null)
        {
            html.append("<p><img src='").append(appLogoURL.toExternalForm()).
                append("'></p>\n");
        }
        html.append("<p>Copyright  ").append(AppInfo.BUILD_YEAR).
            append(" by Dirk S. Grossmann</p>");
        html.append("</td></tr>\n</table>");
        html.append("</td></tr>\n</table>");
        html.append("</body>\n</html>");
        return html.toString();
    } // showOff
} // ContentsPreviewManager
