/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: DirTreeModel
    Created: 2 January, 2003
        $Id: DirTreeModel.java 159 2009-05-19 19:40:47Z dirk $
  $Revision: 159 $
      $Date: 2009-05-19 21:40:47 +0200 (Di, 19 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.dir.SeriesContainer;
import com.dgrossmann.photo.dir.persist.PersistException;
import com.dgrossmann.photo.ui.ExplorerMainFrame;

/**
 * Implements the tree model for the series directory tree.
 */
public class DirTreeModel implements TreeModel
{
    private ExplorerMainFrame       m_mainFrame;
    private List<TreeModelListener> m_treeModelListeners;
    private SeriesContainer         m_seriesContainer;
    private FileObject              m_root;
    private DirectoryObject[]       m_seriesDirs;

    /**
     * Creates a new <tt>DirTreeModel</tt> instance.
     * @param seriesContainer - Series container object to represent
     * @param parent - Parent component in the user interface
     */
    public DirTreeModel
        ( SeriesContainer   seriesContainer
        , ExplorerMainFrame parent
        )
    {
        m_seriesContainer = seriesContainer;
        m_mainFrame = parent;
        m_treeModelListeners = new ArrayList<TreeModelListener>();
        m_root = new FileObject("Series Directories", null);
        this.refresh(null);
    } // DirTreeModel

    /**
     * Refreshes the tree beginning from the node.
     * @param node - Root node of the subtree to be refreshed.
     */
    public void refresh (AbstractFSObject node)
    {
        if (node == null)
        {
            m_seriesDirs = m_seriesContainer.getSeriesDirectories();
            this.fireTreeStructureChanged(null);
            return;
        }
        this.fireTreeStructureChanged(node);
    } // refresh

    /**
     * Fires a tree structure change for the object.
     * @param node - Root node of the changed subtree
     */
    public void fireTreeStructureChanged (AbstractFSObject node)
    {
    	List<AbstractFSObject> nodes;
        Object[]               path;
        TreeModelEvent         event;
        int                    i, j, len;

        len = m_treeModelListeners.size();
        if (node == null)
            path = new Object[]{m_root};
        else
        {
            // Create the path of the object.
            nodes = new ArrayList<AbstractFSObject>();
            while (node.getParent() != null)
            {
                nodes.add(node);
                node = node.getParent();
            }
            nodes.add(node);
            nodes.add(m_root);
            path = new Object[nodes.size()];
            j = 0;
            for (i = nodes.size() - 1; i >= 0; i--)
                path[j++] = nodes.get(i);
        }
        event = new TreeModelEvent(this, path);
        for (i = 0; i < len; i++)
            m_treeModelListeners.get(i).treeStructureChanged(event);
    } // fireTreeStructureChanged

    //-------------------------------------------------------------------------
    //  TreeModel interface methods
    //=========================================================================

    /**
     * Adds a listener for the TreeModelEvent posted after the tree changes.
     * @param l - Tree nodel listener to add
     */
    public void addTreeModelListener (TreeModelListener l)
    {
        m_treeModelListeners.add(l);
    } // addTreeModelListener

    /**
     * Removes a listener previously added with addTreeModelListener().
     * @param l - Tree nodel listener to remove
     */
    public void removeTreeModelListener (TreeModelListener l)
    {
        m_treeModelListeners.remove(l);
    } // removeTreeModelListener

    /**
     * Returns the child of parent at index index in the parent's child array.
     * @param parent - Parent node
     * @param index - <tt>0</tt>-based child node index
     * @return Child node at the specified index
     */
    public Object getChild (Object parent, int index)
    {
        // Is it the root (a misused file) ?-
        if (parent instanceof FileObject)
            return m_seriesDirs[index];
        // Ordinary directory.
        DirectoryObject de = (DirectoryObject) parent;
        return de.getSubDirAt(index);
    } // getChild

    /**
     * Returns the number of children of parent.
     * @param parent - Parent node
     * @return The child count
     */
    public int getChildCount (Object parent)
    {
        // Is it the root (a misused file) ?-
        if (parent instanceof FileObject)
            return m_seriesDirs.length;
        // Ordinary directory.
        DirectoryObject de = (DirectoryObject) parent;
        return de.getSubDirCount();
    } // getChildCount

    /**
     * Returns the index of child in parent.
     * @param parent - Parent node
     * @param child - Child node
     * @return <tt>0</tt>-based index of the child node
     */
    public int getIndexOfChild (Object parent, Object child)
    {
        DirectoryObject ch = (DirectoryObject) child;
        // Is it the root (a misused file) ?-
        if (parent instanceof FileObject)
        {
            for (int i = 0; i < m_seriesDirs.length; i++)
            {
                if (ch == m_seriesDirs[i])
                    return i;
            }
            return -1;
        }
        // Ordinary directory.
        DirectoryObject de = (DirectoryObject) parent;
        return de.getIndexOfSubDir(ch);
    } // getIndexOfChild

    /**
     * Returns the root of the tree.
     * @return Root node object
     */
    public Object getRoot ()
    {
        return m_root;
    } // getRoot

    /**
     * Returns true if node is a leaf.
     * @param node - Node object to be checked
     * @return <tt>True</tt> iff this node is a leaf
     */
    public boolean isLeaf (Object node)
    {
        // We are all nodes.
        return false;
    } // isLeaf

    /**
     * Gets the directory object for a path in the directory tree.
     * @param treePath - Tree path object
     * @param bExact - <tt>True</tt> if an exact match is needed, <tt>false</tt>
     * if an ancestor of the node denoted by the tree path is sufficient
     * @return Directory object denoted by the tree path or <tt>null</tt> if
     * there is no such object
     */
    public DirectoryObject getDirObjectForPath
        ( TreePath treePath
        , boolean  bExact
        )
    {
    	Iterator<DirectoryObject> subDirIter;
        DirectoryObject           dirObj, d;
        Object[]                  path;
        String                    str;
        int                       i;

        if (treePath == null)
            return null;
        // Find the directory for this path.
        dirObj = null;
        path = treePath.getPath();
        if (path.length >= 2)
        {
            str = path[1].toString();
            for (i = 0; i < m_seriesDirs.length; i++)
            {
                if (m_seriesDirs[i].getFileName().equalsIgnoreCase(str))
                {
                    dirObj = m_seriesDirs[i];
                    break;
                }
            }
        }
        for (i = 2; dirObj != null && i < path.length; i++)
        {
            str = path[i].toString();
            subDirIter = dirObj.getSubDirIterator();
            if (bExact)
                dirObj = null;
            while (subDirIter.hasNext())
            {
                d = subDirIter.next();
                if (d.getFileName().equalsIgnoreCase(str))
                {
                    dirObj = d;
                    break;
                }
            }
        }
        return dirObj;
    } // getDirObjectForPath

    /**
     * Messaged when the user has altered the value for the item
     * identified by path to newValue.
     * @param treePath - Path to the changed node
     * @param newValue - New value
     */
    public void valueForPathChanged (TreePath treePath, Object newValue)
    {
        DirectoryObject dirObj;
        String          str;

        // Find the directory for this path.
        dirObj = this.getDirObjectForPath(treePath, true);
        if (dirObj == null || newValue == null)
            return;
        str = newValue.toString().trim();
        // Does a directory of the desired name already exist ?-
        if (dirObj.getParent() != null)
        {
            File f = new File(dirObj.getParent().getFileName(), str);
            if (f.exists())
            {
                JOptionPane.showMessageDialog(m_mainFrame,
                    "Cannot rename as a directory or file with the following "
                    + "name already exists:\n" + f.getAbsolutePath(),
                    AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
                return;
            }
        }
        // No, continue with renaming.
        if (m_mainFrame != null)
            m_mainFrame.renameExportedFiles(dirObj, null, str);
        dirObj.setFileName(str, true);
        try
        {
            m_seriesContainer.saveSeries(dirObj, true);
        }
        catch (PersistException pe)
        {
            // Show the save error dialog.
            JOptionPane.showMessageDialog(m_mainFrame, pe.getMessage(),
                AppInfo.APP_NAME, JOptionPane.ERROR_MESSAGE);
        }
        // Fire refresh event.
        m_mainFrame.getDirContentsPanel().refresh();
    } // valueForPathChanged
} // DirTreeModel
