/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: AbstractExplorerPanel
    Created: 18 December, 2005
        $Id: AbstractExplorerPanel.java 159 2009-05-19 19:40:47Z dirk $
  $Revision: 159 $
      $Date: 2009-05-19 21:40:47 +0200 (Di, 19 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.panel;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.JPanel;

import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.ui.ExplorerMainFrame;

/**
 * Abstract base class for panels in the Explorer main frame.
 * @author Dirk Grossmann
 */
public abstract class AbstractExplorerPanel extends JPanel
    implements IExplorerPanel
{
    private ExplorerMainFrame                  m_frame;
    private PanelTitleComponent                m_panelTitleComponent;
    private List<IExplorerPanelChangeListener> m_changeListeners;
    private DirectoryObject                    m_currentDirectory;
    private List<AbstractFSObject>             m_currentFsObjects;

    /**
     * Creates a new <tt>AbstractExplorerPanel</tt> instance.
     * @param frm
     * @param initialTitleText
     */
    public AbstractExplorerPanel
        ( ExplorerMainFrame frm
        , String            initialTitleText
        )
    {
        m_frame = frm;
        m_panelTitleComponent = new PanelTitleComponent(initialTitleText);
        m_panelTitleComponent.setInactiveColors();
        m_changeListeners = new ArrayList<IExplorerPanelChangeListener>(2);
        m_currentDirectory = null;
        m_currentFsObjects = null;
    } // AbstractExplorerPanel

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#getRootComponent()
     */
    public JPanel getRootComponent ()
    {
        return this;
    } // getRootComponent

    /**
     * Gets the titleLabel value.
     * @return The titleLabel
     */
    public PanelTitleComponent getTitleLabel ()
    {
        return m_panelTitleComponent;
    } // getTitleLabel

    /**
     * Gets the frame value.
     * @return The frame
     */
    public ExplorerMainFrame getFrame ()
    {
        return m_frame;
    } // getFrame

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#addChangeListener(com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener)
     */
    public void addChangeListener (IExplorerPanelChangeListener listener)
    {
        m_changeListeners.add(listener);
    } // addChangeListener

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#removeChangeListener(com.dgrossmann.photo.ui.panel.IExplorerPanelChangeListener)
     */
    public void removeChangeListener (IExplorerPanelChangeListener listener)
    {
        m_changeListeners.remove(listener);
    } // removeChangeListener

    /**
     * Sets the title of the panel.
     * @param title - The new panel title
     */
    public void setTitle (String title)
    {
        m_panelTitleComponent.setText((title == null || title.length() == 0) ?
            " " : title);
    } // setTitle

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setActiveColors()
     */
    public void setActiveColors ()
    {
        m_panelTitleComponent.setActiveColors();
    } // setActiveColors

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setInactiveColors()
     */
    public void setInactiveColors ()
    {
        m_panelTitleComponent.setInactiveColors();
    } // setInactiveColors

    /**
     * Gets the currentDirectory value.
     * @return The currentDirectory
     */
    public DirectoryObject getCurrentDirectory ()
    {
        return m_currentDirectory;
    } // getCurrentDirectory

    /**
     * Sets a new currentDirectory value.
     * @param currentDirectory - The currentDirectory to set
     */
    public void setCurrentDirectory (DirectoryObject currentDirectory)
    {
        m_currentDirectory = currentDirectory;
    } // setCurrentDirectory

    /**
     * Gets the currentFsObjects value.
     * @return The currentFsObjects
     */
    public List<AbstractFSObject> getCurrentFSObjects ()
    {
        return m_currentFsObjects;
    } // getCurrentFSObjects

    /**
     * Sets a new currentFsObjects value.
     * @param currentFsObjects - The currentFsObjects to set
     */
    public void setCurrentFSObjects (List<AbstractFSObject> currentFsObjects)
    {
        m_currentFsObjects = currentFsObjects;
    } // setCurrentFSObjects

    /**
     * @see com.dgrossmann.photo.ui.panel.IExplorerPanel#setCurrentFSObject(com.dgrossmann.photo.dir.AbstractFSObject)
     */
    public void setCurrentFSObject (AbstractFSObject fsObj)
    {
        List<AbstractFSObject> objs = new ArrayList<AbstractFSObject>(1);
        objs.add(fsObj);
        this.setCurrentFSObjects(objs);
    } // setCurrentFSObject

    /**
     * Fires a current directory changed event.
     * @param currentDirObj - The new current directory
     */
    public void fireCurrentDirectoryChanged (DirectoryObject currentDirObj)
    {
        Iterator<IExplorerPanelChangeListener> iter;
        iter = m_changeListeners.iterator();
        while (iter.hasNext())
            iter.next().onCurrentDirectoryChanged(this, currentDirObj);
    } // fireCurrentDirectoryChanged

    /**
     * Fires a selected objects changed event
     * @param currentFsObjects - List of the currently selected file system
     * objects
     */
    public void fireSelectionChanged (List<AbstractFSObject> currentFsObjects)
    {
        Iterator<IExplorerPanelChangeListener> iter;
        iter = m_changeListeners.iterator();
        while (iter.hasNext())
            iter.next().onSelectionChanged(this, currentFsObjects);
    } // fireSelectionChanged

    /**
     * Fires a selected objects changed event
     * @param fsObj - List of the currently selected file system
     * objects
     */
    public void firePropertiesChanged (AbstractFSObject fsObj)
    {
        Iterator<IExplorerPanelChangeListener> iter;
        iter = m_changeListeners.iterator();
        while (iter.hasNext())
            iter.next().onPropertiesChanged(this, fsObj);
    } // firePropertiesChanged
} // AbstractExplorerPanel
