/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: AssociateFileTableModel
    Created: 2 January, 2003
        $Id: AssociateFileTableModel.java 160 2009-05-31 07:57:29Z dirk $
  $Revision: 160 $
      $Date: 2009-05-31 09:57:29 +0200 (So, 31 Mai 2009) $
    $Author: dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Component;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.ui.ImageHolder;
import com.dgrossmann.photo.ui.panel.contents.ContentsTableModel;

/**
 * Implements the table model for the <i>Associate with File</i> dialog. Unlike
 * its base class, it only delivers <tt>FileObject</tt>'s that are files (no
 * references).
 */
public class AssociateFileTableModel extends ContentsTableModel
{
    private static final String[] s_columnNames =
    {
        "", "File Name", "Titel", "Modified"
    };
    private static final Class<?>[] s_columnTypes =
    {
        ImageIcon.class, String.class, String.class, String.class
    };

    private List<FileObject> m_files;

    /**
     * Creates a new <tt>AssociateFileTableModel</tt> instance.
     * @param parent - Parent component needed by the image holder
     * @param imgHolder - Image holder object that is responsible for image
     * loading and caching
     */
    public AssociateFileTableModel (Component parent, ImageHolder imgHolder)
    {
        super(parent, null, imgHolder);
        m_bWithImagePreview = false;
        m_files = null;
    } // AssociateFileTableModel

    /**
     * Private helper to set the list of files to be delivered from the
     * "dirObj" directory object.
     * @param dirObj - The directory object
     */
    private void setFileList (DirectoryObject dirObj)
    {
        FileObject fileObj;
        int        fileCount, i;

        fileCount = dirObj.getFileCount();
        m_files = new ArrayList<FileObject>(fileCount);
        for (i = 0; i < fileCount; i++)
        {
            fileObj = dirObj.getFileAt(i);
            if (!fileObj.isReference())
                m_files.add(fileObj);
        }
    } // setFileList

    /**
     * Sets the current directory object.
     * @see com.dgrossmann.photo.ui.panel.contents.ContentsTableModel#setCurrentDir(com.dgrossmann.photo.dir.DirectoryObject)
     */
    @Override
	public void setCurrentDir (DirectoryObject currentDir)
    {
        this.setFileList(currentDir);
        super.setCurrentDir(currentDir);
    } // setCurrentDir

    /**
     * Refreshes the table.
     * @see com.dgrossmann.photo.ui.panel.contents.ContentsTableModel#refresh()
     */
    @Override
	public void refresh ()
    {
        this.setFileList(m_currentDir);
        super.refresh();
    } // refresh

    /**
     * Gets the column count.
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
	public int getColumnCount ()
    {
        return s_columnNames.length;
    } // getColumnCount

    /**
     * Gets the name of the column identified by index.
     * @see javax.swing.table.TableModel#getColumnName(int)
     */
    @Override
	public String getColumnName (int index)
    {
        return s_columnNames[index];
    } // getColumnName

    /**
     * Gets the class of the column identified by index.
     * @see javax.swing.table.TableModel#getColumnClass(int)
     */
    @Override
	public Class<?> getColumnClass (int index)
    {
        return s_columnTypes[index];
    } // getColumnClass

    /**
     * Gets whether a cell is editable.
     * @see javax.swing.table.TableModel#isCellEditable(int, int)
     */
    @Override
	public boolean isCellEditable (int row, int column)
    {
        return false;
    } // isCellEditable

    /**
     * Gets the row count.
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
	public int getRowCount ()
    {
        return (m_files != null) ? m_files.size() : 0;
    } // getRowCount

    /**
     * Gets the file object at the specified index.
     * @param index - The index
     * @return The file object
     */
    public FileObject getFileAt (int index)
    {
        if (m_files == null || index < 0 || index >= m_files.size())
            return null;
        return m_files.get(index);
    } // getFileAt

    /**
     * Gets the cell value at (row, column).
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
	public Object getValueAt (int row, int column)
    {
        FileObject fe = this.getFileAt(row);
        if (fe == null)
            return "";
        switch (column)
        {
            case 0: // File type image.
                return this.getTypeImage(fe);
            case 1: // File name.
                return fe.getFileName();
            case 2: // File title.
                if (fe.isSeparator())
                    return "---------";
                return fe.getTitle(true);
            case 3: // Date
                return fe.getModDateTimeString(true);
        }
        return "";
    } // getValueAt
} // AssociateFileTableModel
