/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2006 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
      Class: AssociateDialog
    Created: 2 January, 2003
        $Id: AssociateDialog.java 42 2006-01-21 17:47:48Z Dirk $
  $Revision: 42 $
      $Date: 2006-01-21 18:47:48 +0100 (Sa, 21 Jan 2006) $
    $Author: Dirk $
===============================================================================
*/

package com.dgrossmann.photo.ui.dialog;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.Toolkit;

import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import com.dgrossmann.photo.AppInfo;
import com.dgrossmann.photo.dir.AbstractFSObject;
import com.dgrossmann.photo.dir.DirectoryObject;
import com.dgrossmann.photo.dir.FileObject;
import com.dgrossmann.photo.ui.ImageHolder;

/**
 * The <i>Associate File with Reference</i> dialog class.
 */
public class AssociateDialog extends JDialog
{
    private static int SZ_MODIFIED_WIDTH_S = 110;
    private static int SZ_MODIFIED_WIDTH_L = 128;

    private ImageHolder m_imgHolder;
    private FileObject  m_currentFile;
    private boolean     m_isChanged;

    /**
     * Creates a new <tt>AssociateDialog</tt> form instance.
     * @param parent - Parent frame in the user interface
     * @param imgHolder - Image holder object that is responsible for image
     * loading and caching
     * @param currentFile - Reference object to be associated with a file
     * @param modal - <tt>True</tt> for a modal dialog
     */
    public AssociateDialog
        ( Frame       parent
        , ImageHolder imgHolder
        , FileObject  currentFile
        , boolean     modal
        )
    {
        super(parent, modal);

        Toolkit   tk = Toolkit.getDefaultToolkit();
        Dimension screenSize = tk.getScreenSize();

        m_imgHolder = imgHolder;
        m_currentFile = currentFile;
        m_isChanged = false;
        this.initComponents();
        listTable.setShowHorizontalLines(true);
        listTable.setGridColor(new Color(222, 222, 211));
        AssociateFileTableModel tModel = (AssociateFileTableModel)
            listTable.getModel();
        tModel.setCurrentDir((DirectoryObject) m_currentFile.getParent());
        if (tModel.getRowCount() == 0)
            okButton.setEnabled(false);
        // Set the table columns.
        JTableHeader thead = listTable.getTableHeader();
        Dimension thSize = thead.getPreferredSize();
        if (thSize.height < 18)
            thSize.height = 18;
        thead.setPreferredSize(thSize);
        listTable.setRowHeight(34);
        listTable.addMouseListener(new AssocTableMouseListener(this));
        TableColumnModel cModel = listTable.getColumnModel();
        TableColumn col;
        // The "Type" column.
        col = cModel.getColumn(0);
        col.setPreferredWidth(32);
        col.setWidth(32);
        col.setMinWidth(32);
        col.setMaxWidth(32);
        col.setResizable(false);
        // The "File Name" column.
        col = cModel.getColumn(1);
        col.setPreferredWidth(180);
        col.setWidth(180);
        col.setMinWidth(20);
        col.setMaxWidth(1000);
        // The "Modified" column.
        int colWidth = (screenSize.width > 1200) ?
            SZ_MODIFIED_WIDTH_L : SZ_MODIFIED_WIDTH_S;
        col = cModel.getColumn(3);
        col.setPreferredWidth(colWidth);
        col.setWidth(colWidth);
        col.setMinWidth(50);
        col.setMaxWidth(200);
        // Set the title texts.
        titleLabel.setText("This dialog associates the reference \""
            + m_currentFile.getTitle(true)
            + "\" with a file in the same directory.");
        String str = m_currentFile.getTitle(true);
        if (str.length() > 0)
        {
            str += " - " + this.getTitle();
            this.setTitle(str);
        }
        // Position centered in the parent dialog.
        Rectangle parentRect, thisRect;
        parentRect = parent.getBounds();
        thisRect = this.getBounds();
        thisRect.x = parentRect.x + (parentRect.width - thisRect.width) / 2;
        thisRect.y = parentRect.y + (parentRect.height - thisRect.height) / 2;
        if (thisRect.x < 0)
            thisRect.x = 0;
        if (thisRect.y < 0)
            thisRect.y = 0;
        this.setBounds(thisRect);
    } // AssociateDialog

    /**
     * Shows the dialog window.
     * @return <tt>True</tt> if the user made changes, <tt>false</tt> if she
     * canceled the dialog
     */
    public boolean showDialog()
    {
        this.setVisible(true);
        return m_isChanged;
    } // showDialog

    /**
     * Invokes the action of the <i>OK</i> button (called by the table mouse
     * listener on double-click).
     */
    public void invokeOkButtonAction ()
    {
        this.okButtonActionPerformed(null);
    } // invokeOkButtonAction

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        titleLabel = new javax.swing.JLabel();
        descLabel = new javax.swing.JLabel();
        desc2Label = new javax.swing.JLabel();
        listScrollPane = new javax.swing.JScrollPane();
        listTable = new javax.swing.JTable();
        buttonPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setTitle("Associate Reference with File");
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

        titleLabel.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        getContentPane().add(titleLabel, gridBagConstraints);

        descLabel.setText("The data from the reference are merged with those of the file, and the reference is deleted afterwards. ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        getContentPane().add(descLabel, gridBagConstraints);

        desc2Label.setText("Please select the file to associate the reference with from the following list:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        getContentPane().add(desc2Label, gridBagConstraints);

        listTable.setModel(new AssociateFileTableModel(this, m_imgHolder));
        listTable.setShowHorizontalLines(false);
        listTable.setShowVerticalLines(false);
        listScrollPane.setViewportView(listTable);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 100.0;
        gridBagConstraints.weighty = 100.0;
        gridBagConstraints.insets = new java.awt.Insets(6, 6, 0, 6);
        getContentPane().add(listScrollPane, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(okButton);

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        buttonPanel.add(cancelButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 6, 0);
        getContentPane().add(buttonPanel, gridBagConstraints);

        pack();
    }
    // </editor-fold>//GEN-END:initComponents

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_okButtonActionPerformed
    {//GEN-HEADEREND:event_okButtonActionPerformed
        FileObject toAssoc = null;
        String     sRef, sFile;
        int        index;

        index = listTable.getSelectedRow();
        if (index >= 0)
        {
            AssociateFileTableModel tModel =
                (AssociateFileTableModel) listTable.getModel();
            toAssoc = tModel.getFileAt(index);
        }
        if (toAssoc == null)
        {
            JOptionPane.showMessageDialog
                (this, "Please select the file you wish to associate "
                + "reference \"" + m_currentFile.getTitle(true)
                + "\"\nwith from the file list.",
                AppInfo.APP_NAME, JOptionPane.WARNING_MESSAGE);
            return;
        }
        // Perform the association. At first we copy all properties of the
        // reference into the file.
        sRef = m_currentFile.getTitle(false);
        if (sRef.length() > 0)
            toAssoc.set(AbstractFSObject.TITLE, sRef);
        sRef = m_currentFile.get(AbstractFSObject.SUBTITLE);
        if (sRef.length() > 0)
            toAssoc.set(AbstractFSObject.SUBTITLE, sRef);
        sRef = m_currentFile.get(AbstractFSObject.HREF);
        if (sRef.length() > 0)
            toAssoc.set(AbstractFSObject.HREF, sRef);
        // Merge the descriptions.
        sRef = m_currentFile.get(AbstractFSObject.DESCRIPTION);
        sFile = toAssoc.get(AbstractFSObject.DESCRIPTION);
        if (sRef.length() > 0)
        {
            if (sFile.length() > 0)
                sRef += "\n\n" + sFile;
            toAssoc.set(AbstractFSObject.DESCRIPTION, sRef);
        }
        sRef = m_currentFile.get(AbstractFSObject.REMARK);
        if (sRef.length() > 0)
            toAssoc.set(AbstractFSObject.REMARK, sRef);
        // Now remove the reference.
        ((DirectoryObject) toAssoc.getParent()).removeFile(m_currentFile);
        m_currentFile = null;
        // Close the dialog.
        m_isChanged = true;
        this.cancelButtonActionPerformed(evt);
    }//GEN-LAST:event_okButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt)//GEN-FIRST:event_cancelButtonActionPerformed
    {//GEN-HEADEREND:event_cancelButtonActionPerformed
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt)
    {//GEN-FIRST:event_closeDialog
        this.setVisible(false);
        this.dispose();
    }//GEN-LAST:event_closeDialog
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JLabel desc2Label;
    private javax.swing.JLabel descLabel;
    private javax.swing.JScrollPane listScrollPane;
    private javax.swing.JTable listTable;
    private javax.swing.JButton okButton;
    private javax.swing.JLabel titleLabel;
    // End of variables declaration//GEN-END:variables
    
} // AssociateDialog
