<?php
/*
-------------------------------------------------------------------------------
  J  P h o t o - E x p l o r e r

  Copyright (c) 2011 by Dirk S. Grossmann.  All rights reserved.
-------------------------------------------------------------------------------
  Desclares the series loader class that loads the series, group, and file
  meta data from a file.
-------------------------------------------------------------------------------
  $Id: img-loader.php 180 2010-12-11 12:40:43Z dirk $
===============================================================================
*/

error_reporting(E_ERROR | E_WARNING | E_PARSE);

/*
-------------------------------------------------------------------------------
  V a r i a b l e   I n i t i a l i z a t i o n s
===============================================================================
*/

// Name of the directory containing this file and the series directories.
$g_seriesBaseDir = $_SERVER['DOCUMENT_ROOT'] . '/photo';

// Constant definitions: standard names.
define("REFERENCE"        , "reference:");
define("SEPARATOR"        , "separator:");

// XML element names.
define("ELEM_SERIES"      , "series");
define("ELEM_GROUP"       , "group");
define("ELEM_CONTENTS"    , "contents");
define("ELEM_SEPARATOR"   , "separator");
define("ELEM_FILE"        , "file");
define("ELEM_TITLE"       , "title");
define("ELEM_SUBTITLE"    , "subtitle");
define("ELEM_LOCATION"    , "location");
define("ELEM_DESCRIPTION" , "description");
define("ELEM_REMARK"      , "remark");
define("ELEM_PROPERTY"    , "property");
// XML attribute names.
define("ATTR_NAME"        , "name");
define("ATTR_REFERENCE"   , "is-reference");
define("ATTR_HREF"        , "href");
define("ATTR_BEGIN_DATE"  , "begin-date");
define("ATTR_END_DATE"    , "end-date");
define("ATTR_WEB_EXPORTED", "web-exported");

// Constant definitions: Metadata file tag names.
define("FOR_WEB"          , "web");
define("SERIES"           , "series");
define("GROUP"            , "group");
define("FILE"             , "file");
define("END"              , "end");
define("END_FILE"         , "end-file");
define("END_DESCRIPTION"  , "end-description");

/*
-------------------------------------------------------------------------------
  Class:    "SeriesLoader"
  Function: Loads the metadata from a file into a series object.
===============================================================================
*/

class SeriesLoader
{
    var $bUseNewDom;
    
    function SeriesLoader ()
    {
        $pos = strpos(phpversion(), '4.', 0);
        $this->bUseNewDom = ($pos === false || $pos != 0);
    } // set constructor

    /* Main interface method to load the metadata into a series. */
    function loadSeries (&$seriesDirObj)
    {
        global $g_seriesBaseDir;
        // Load the XML metadata file.
        $dataFileName = $g_seriesBaseDir . '/' . $seriesDirObj->fileName
            . '/_' . $seriesDirObj->fileName . '-metadata.xml';
        if (file_exists($dataFileName))
        {
            $this->loadXmlMetadataFile($seriesDirObj, $dataFileName);
            return;
        }
        // Now try the old text format.
        $dataFileName = $g_seriesBaseDir . '/' . $seriesDirObj->fileName
            . '/_' . $seriesDirObj->fileName . '_metadata.txt';
        $fd = fopen($dataFileName, 'r');
        if (!$fd)
        {
            echo '<p><b>Error</b>: Cannot open metadata file:<br>'
                . "&quot;$dataFileName&quot;</p>";
            return;
        }
        $this->loadTextMetadataFile($seriesDirObj, $fd);
        fclose($fd);
    } // loadSeries

    /*
    ---------------------------------------------------------------------------
      XML metadata loading functions
    ===========================================================================
    */

    /* Main method to load an XML metadata document. */
    function loadXmlMetadataFile (&$seriesDirObj, $dataFileName)
    {
        // Load the XML document.
        if ($this->bUseNewDom)
        {
            $doc = & new DOMDocument();
            $doc->encoding = "ISO-8859-1";
            $doc->load($dataFileName);
        }
        else
            $doc = domxml_open_file($dataFileName);
        // Test the XML document.
        if (!$doc)
        {
            echo '<p><b>Error</b>: Error while parsing the metadata file:<br>'
                . "&quot;$dataFileName&quot;</p>";
            return;
        }
        $rootElem = ($this->bUseNewDom) ?
            $doc->documentElement : $doc->document_element();
        $tagName = ($this->bUseNewDom) ?
            $rootElem->tagName :
            $rootElem->tagname();
        if (strcmp($tagName, ELEM_SERIES))
        {
            echo '<p><b>Error</b>: Metadata file is not a JPhoto-Explorer '
                . 'series meta data file:<br>'
                . "&quot;$dataFileName&quot;</p>";
            return;
        }
        $hasWebExport = ($this->bUseNewDom) ?
            $rootElem->hasAttribute(ATTR_WEB_EXPORTED) :
            $rootElem->has_attribute(ATTR_WEB_EXPORTED);
        if (!$hasWebExport)
        {
            echo '<p><b>Error</b>: Metadata file is not Web-exported:<br>'
                . "&quot;$dataFileName&quot;</p>";
            return;
        }
        // Read the series description and the groups.
        $this->readDescription($seriesDirObj, $rootElem);
        $this->readGroup($seriesDirObj, $seriesDirObj, $rootElem);
    } // loadXmlMetadataFile

    /* Reads the meta data for a series or group. */
    function readGroup (&$seriesDirObj, &$parentDirObj, &$dirElement)
    {
        // Get the contents element.
        $contentsElement = null;
        $subNodeArray = ($this->bUseNewDom) ? $dirElement->childNodes :
            $dirElement->child_nodes();
        foreach ($subNodeArray as $subNode)
        {
            $nodeType = ($this->bUseNewDom) ? $subNode->nodeType :
                $subNode->node_type();
            if ($nodeType != XML_ELEMENT_NODE)
                continue;
            $tagName = ($this->bUseNewDom) ?
                $subNode->tagName :
                $subNode->tagname();
            if (!strcmp($tagName, ELEM_CONTENTS))
            {
                $contentsElement = &$subNode;
                break;
            }
        }
        if (!$contentsElement)
            return;
        // Get the sub-elements of the contents.
        $subNodeArray = ($this->bUseNewDom) ? $contentsElement->childNodes :
            $contentsElement->child_nodes();
        foreach ($subNodeArray as $subNode)
        {
            $nodeType = ($this->bUseNewDom) ? $subNode->nodeType :
                $subNode->node_type();
            if ($nodeType != XML_ELEMENT_NODE)
                continue;
            $tagName = ($this->bUseNewDom) ?
                    $subNode->tagName :
                    $subNode->tagname();
            // Create the sub-fs-objects by tag name: group, file, separator.
            if (!strcmp($tagName, ELEM_GROUP))
            {
                // This is a group (directory).
                $dirObj = & new DirectoryObject("", $parentDirObj);
                $seriesDirObj->addGroupInSeries($dirObj);
                $this->readDescription($dirObj, $subNode);
                // Read the group contents.
                $this->readGroup($seriesDirObj, $dirObj, $subNode);
            }
            else if (!strcmp($tagName, ELEM_FILE) ||
                     !strcmp($tagName, ELEM_SEPARATOR))
            {
                // This is a file or reference.
                $fileObj = & new FileObject("", $parentDirObj);
                $this->readDescription($fileObj, $subNode);
                if (!strcmp($tagName, ELEM_SEPARATOR))
                    $fileObj->fileName = SEPARATOR;
            }
        }
    } // readGroup

    /*
     * Reads a description from the XML element and fills the file system
     * object.
     */
    function readDescription (&$fsObj, &$fsElement)
    {
        // Get the name.
        $attrib = ($this->bUseNewDom) ?
            $fsElement->getAttributeNode(ATTR_NAME) :
            $fsElement->get_attribute_node(ATTR_NAME);
        if ($attrib)
            $fsObj->fileName = $attrib->value;
        // Get the begin and end date.
        $attrib = ($this->bUseNewDom) ?
            $fsElement->getAttributeNode(ATTR_BEGIN_DATE) :
            $fsElement->get_attribute_node(ATTR_BEGIN_DATE);
        if ($attrib)
            $fsObj->set(DATE_BEGIN, $attrib->value);
        $attrib = ($this->bUseNewDom) ?
            $fsElement->getAttributeNode(ATTR_END_DATE) :
            $fsElement->get_attribute_node(ATTR_END_DATE);
        if ($attrib)
            $fsObj->set(DATE_END, $attrib->value);
        // Get whether it is a reference.
        $hasReference = ($this->bUseNewDom) ?
            $fsElement->hasAttribute(ATTR_REFERENCE) :
            $fsElement->has_attribute(ATTR_REFERENCE);
        if ($hasReference)
        {
            $this->fileName = REFERENCE;
            $attrib = ($this->bUseNewDom) ?
            $fsElement->getAttributeNode(ATTR_HREF) :
            $fsElement->get_attribute_node(ATTR_HREF);
            if ($attrib)
                $fsObj->set(HREF, $attrib->value);
        }
        // Get the description strings.
        $subNodeArray = ($this->bUseNewDom) ? $fsElement->childNodes :
            $fsElement->child_nodes();
        foreach ($subNodeArray as $subNode)
        {
            $nodeType = ($this->bUseNewDom) ? $subNode->nodeType :
                $subNode->node_type();
            if ($nodeType != XML_ELEMENT_NODE)
                continue;
            $tagName = ($this->bUseNewDom) ?
                    $subNode->tagName :
                    $subNode->tagname();
            if (!strcmp($tagName, ELEM_TITLE))
                $fsObj->set(TITLE, $this->getTextContent($subNode));
            else if (!strcmp($tagName, ELEM_SUBTITLE))
                $fsObj->set(SUBTITLE, $this->getTextContent($subNode));
            else if (!strcmp($tagName, ELEM_DESCRIPTION))
                $fsObj->set(DESCRIPTION, $this->getTextContent($subNode));
            else if (!strcmp($tagName, ELEM_LOCATION))
                $fsObj->set(LOCATION, $this->getTextContent($subNode));
            else if (!strcmp($tagName, ELEM_REMARK))
                $fsObj->set(REMARK, $this->getTextContent($subNode));
            else if (!strcmp($tagName, ELEM_PROPERTY))
            {
                $name = ($this->bUseNewDom) ?
                    $subNode->getAttribute(ATTR_NAME) :
                    $subNode->get_attribute(ATTR_NAME);
                $fsObj->set($name, $this->getTextContent($subNode));
            }
        }
    } // readDescription

    /* Helper to get the content (including descendants) of a node. */
    function getTextContent (&$node)
    {
        if ($this->bUseNewDom)
            return $node->textContent;
        // May not work.
        return $node->get_content();
    } // getTextContent
    
    /*
    ---------------------------------------------------------------------------
      Old text metadata loading functions
    ===========================================================================
    */

    /* Main method to load an old text metadata document. */
    function loadTextMetadataFile (&$seriesDir, $fd)
    {
        // Read the file contents. Read the @web tag.
        $line = '';
        $arr = $this->readTextValue($fd, $line);
        $line = $arr[2];
        if (strcmp($arr[0], FOR_WEB))
        {
            echo '<p><b>Error</b>: @web expected, not @' . $arr[0] . "</p>\n";
            return;
        }
        $arr = $this->readTextValue($fd, $line);
        $line = $arr[2];
        if (strcmp($arr[0], SERIES))
        {
            echo '<p><b>Error</b>: @series expected, not @'. $arr[0] ."</p>\n";
            return;
        }
        if (!feof($fd))
            $line = $this->readTextDescription($seriesDir, $fd, $line);
        else
            echo '<p><b>Error</b>: Unexpected end-of-file.</p>';
        if (!feof($fd))
            $line = $this->readTextGroup($seriesDir, $seriesDir, $fd, $line);
        else
            echo '<p><b>Error</b>: Unexpected end-of-file.</p>';
    } // loadTextMetadataFile
    
    function readTextValue ($fd, $line)
    {
        // Go to the next '@' line.
        while (strlen($line) == 0 || $line[0] != '@')
        {
            if (feof($fd))
                return array('', '', '');
            $line = trim(fgets($fd, 4096));
            if (strlen($line) == 0 || $line[0] == '#')
                continue;
            if ($line[0] != '@')
            {
                 echo '<p><b>Error</b>: Unexpected line in metadata file: '
                         . $line . "</p>\n";
                 return array('', '', '');
            }
        }
        // Split the "@name val ..." string.
        $line = substr($line, 1);
        $pos = strpos($line, ' ');
        if ($pos === false)
            $name = strtolower($line);
        else
        {
            $name = strtolower(substr($line, 0, $pos));
            $value = trim(substr($line, $pos));
        }
        // Test for an "@end" token.
        if (!strncmp($name, END, strlen(END)))
            return array($name, '', '');
        // The value can encompass the following lines until "@..."
        while (!feof($fd))
        {
            $line = trim(fgets($fd, 4096));
            if ($line[0] == "@")
                return array($name, $value, $line);
            $value .= "\n" . $line;
        }
        return array($name, $value, '');
    } // readTextValue

    function readTextDescription (&$fsObj, $fd, $line)
    {
        while (!feof($fd))
        {
            $arr = $this->readTextValue($fd, $line);
            $name = $arr[0];
            $line = $arr[2];
            if (!strcmp($name, END_DESCRIPTION) || !strcmp($name, END_FILE))
                return $line;
            if (strlen($name) == 0)
                return $line;
            $fsObj->set($name, $arr[1]);
        }
        return '';
    } // readTextDescription

    function readTextGroup (&$seriesDir, &$parentDir, $fd, $line)
    {
        // A block is a sequence of files and blocks, terminated by "@end".
        while (!feof($fd))
        {
            $arr = $this->readTextValue($fd, $line);
            $name = $arr[0];
            $line = $arr[2];
            if (!strcmp($name, END))
                return $line;
            if (!strcmp($name, FILE))
            {
                // This is a file or reference.
                $fileObj = & new FileObject($arr[1], $parentDir);
                $line = $this->readTextDescription($fileObj, $fd, $line);
                continue;
            }
            if (!strcmp($name, GROUP))
            {
                // This is a group (directory).
                $dirObj = & new DirectoryObject($arr[1], $parentDir);
                $seriesDir->addGroupInSeries($dirObj);
                $line = $this->readTextDescription($dirObj, $fd, $line);
                // Read the group contents.
                $line = $this->readTextGroup($seriesDir, $dirObj, $fd, $line);
                continue;
            }
            echo '<p><b>Error</b>: @file, @group or @end expected at (@' .
                $name .' ' . $arr[1] . ') near line: ' . $line . "</p>\n";
            return '';
        }
        return '';
    } // readTextGroup
} // SeriesLoader
?>
